/* mirt.c
 *
 * Copyright (C) 2007 Stephane Germain
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

/**
   \file

   \brief A command line program to estimate items and abilities
   in a multiple choice IRT model.

   \author Stephane Germain <germste@gmail.com>
*/

#include <libirt.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <getopt.h>
#include <string.h>

/**
   \brief The command line options.

   To be used by getopt_long.
*/
struct option long_options[] = {
  {"help", no_argument, 0, 'h'},
  {"version", no_argument, 0, 'v'},
  {"nbr_item", required_argument, 0, 'i'},
  {"nbr_subject", required_argument, 0, 'j'},
  {"skip", required_argument, 0, 'k'},
  {"offset", required_argument, 0, 'o'},
  {"extra", required_argument, 0, 'x'},
  {"delimiter", required_argument, 0, 'd'},
  {"nbr_quad", required_argument, 0, 'q'},
  {"penalized", no_argument, 0, 'P'},
  {"kernel", no_argument, 0, 'K'},
  {"smooth_factor", required_argument, 0, 's'},
  {"max_em_iter", required_argument, 0, 'e'},
  {"max_nr_iter", required_argument, 0, 'n'},
  {"response_functions", required_argument, 0, 'R'},
  {"grouping", no_argument, 0, 'G'},
  {"verbose", no_argument, 0, 'V'},
  {"ability", no_argument, 0, 'A'},
  {"quad_from", required_argument, 0, 11},
  {"quad_to", required_argument, 0, 12},
  {"answer_key", required_argument, 0, 'W'},
  {"trimming", no_argument, 0, 'T'},
  {"slope_init", required_argument, 0, 16},
  {"thresh_init", required_argument, 0, 17},
  {"precision", required_argument, 0, 19},
  {"graded", no_argument, 0, 'g'},
  {"icc", no_argument, 0, 'I'},
  {0, 0, 0, 0}
};

/** \brief The file name. */
char *file_name;

/** \brief The number of item. */
int nbr_item;

/** \brief The number of subject. */
int nbr_subject;

/** \brief The number of line to skip. */
int skip;

/** \brief The number of character or field to skip. */
int offset;

/** \brief The number of character or field between items. */
int extra;

/** \brief The delimiter. */
char delimiter;

/** \brief The option representing sucess for each item. */
char *answer_key;

/** \brief The number of quadrature class. */
int nbr_quad;

/** \brief The first class middle point. */
double quad_from;

/** \brief The last class middle point. */
double quad_to;

/** \brief The initial value of the slope. */
double slope_init;

/** \brief The initial value of the threshold. */
double thresh_init;

/** \brief The maximum number of EM iteration. */
int max_em_iter;

/** \brief The maximum number of newton iteration. */
int max_nr_iter;

/** \brief The precision. */
double precision;

/** \brief Enable the grouping of identical patterns. */
int grouping;

/** \brief The verbosity level. */
int verbose;

/** \brief Enable the estimation of the abilities. */
int ability;

/** \brief Enable the trimming of spaces in the fields (when delimited). */
int trimming;

/** \brief Enable the use of penalized maximum marginal likelihood. */
int penalized;

/** \brief Enable the use of kernel estimators. */
int kernel;

/** \brief The smooth factor if PMMLE or kernel is used. */
double smooth_factor;

/** \brief Enable the display of the response functions instead of the parameters. */
int response_functions;

/** \brief Enable the use of the graded model. */
int graded;

/** \brief Enable the display of the items' characteristic curves. */
int icc;

/**
   \brief Parse the options and arguments of the program.

   Also set the defaults values.

   \return 1 for success and 0 for failure.
*/
int
parse_arg (int argc, char *argv[])
{
  int opt = 0;
  int option_index = 0;

  /* the defaults */
  file_name = NULL;
  nbr_item = 0;
  nbr_subject = 0;
  skip = 0;
  offset = 0;
  extra = 0;
  delimiter = ',';
  answer_key = NULL;
  nbr_quad = 64;
  quad_from = -4.0;
  quad_to = 4.0;
  slope_init = 1;
  thresh_init = 1;
  max_em_iter = 100;
  max_nr_iter = 100;
  precision = 1e-5;
  grouping = 0;
  verbose = 0;
  ability = 0;
  trimming = 0;
  penalized = 0;
  kernel = 0;
  smooth_factor = 0.0;
  response_functions = 0;
  graded = 0;
  icc = 0;

  opterr = 1;

  /* the parsing itself */
  while ((opt =
	  getopt_long (argc, argv, "hi:j:k:o:x:d:q:PKe:n:GVAW:Tvs:gI",
		       long_options, &option_index)) != -1)
    {
      switch (opt)
	{
	case 0:
	  break;
	case 'h':
	  printf ("USAGE : mirt [OPTION] file_name\n");
	  printf ("\n");
	  printf
	    ("Estimate by MMLE (marginal maximum likelihood) the parameters\n");
	  printf
	    ("of each item in an multiple choice IRT (item response theory) model.\n");
	  printf ("\n");
	  printf
	    ("Each line of the input file should contains the responses\n");
	  printf("(positive integers) of a subject to each item.\n");
	  printf("By default the responses are assumed to be comma delimited.\n");
	  printf ("\n");
	  printf ("Consult the manual fore more information.\n");
	  printf ("\n");
	  printf ("Report bugs to <germste@gmail.com>.\n");
	  exit (0);
	  break;
	case 'v':
	  printf ("Using version %s of libirt.\n", libirt_version);
	  printf ("Copyright (C) 2005 Stephane Germain.\n");
	  printf ("libirt comes with NO WARRANTY,\n");
	  printf ("to the extent permitted by law.\n");
	  printf ("You may redistribute copies of libirt\n");
	  printf ("under the terms of the GNU General Public License.\n");
	  printf ("For more information about these matters,\n");
	  printf ("see the file named COPYING.\n");
	  exit (0);
	  break;
	case 'i':
	  nbr_item = atoi (optarg);
	  if (nbr_item < 0)
	    {
	      printf ("nbr_item should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'j':
	  nbr_subject = atoi (optarg);
	  if (nbr_subject < 0)
	    {
	      printf ("nbr_subject should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'k':
	  skip = atoi (optarg);
	  if (skip < 0)
	    {
	      printf ("skip should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'o':
	  offset = atoi (optarg);
	  if (offset < 0)
	    {
	      printf ("offset should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'x':
	  extra = atoi (optarg);
	  if (extra < 0)
	    {
	      printf ("extra should be a non-negative integer\n");
	      return 0;
	    }
	  break;
	case 'd':
	  delimiter = optarg[0];
	  break;
	case 'q':
	  nbr_quad = atoi (optarg);
	  if (nbr_quad < 2)
	    {
	      printf ("nbr_quad should be a integer bigger than 1\n");
	      return 0;
	    }
	  break;
	case 'e':
	  max_em_iter = atoi (optarg);
	  if (max_em_iter < 0)
	    {
	      printf ("max_em_iter should be a non-negative integer.\n");
	      return 0;
	    }
	  break;
	case 'n':
	  max_nr_iter = atoi (optarg);
	  if (max_nr_iter < 0)
	    {
	      printf ("max_nr_iter should be a non-negative integer.\n");
	      return 0;
	    }
	  break;
	case 'p':
	  response_functions = 1;
	  break;
	case 'G':
	  grouping = 1;
	  break;
	case 'V':
	  verbose++;
	  break;
	case 'A':
	  ability = 1;
	  break;
	case 11:
	  quad_from = atof (optarg);
	  break;
	case 12:
	  quad_to = atof (optarg);
	  break;
	case 'W':
	  answer_key = optarg;
	  break;
	case 'T':
	  trimming = 1;
	  break;
	case 'P':
	  penalized = 1;
	  break;
	case 'K':
	  kernel = 1;
	  break;
	case 's':
	  smooth_factor = atof (optarg);
	  if (smooth_factor <= 0)
	    {
	      printf ("smooth_factor should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 16:
	  slope_init = atof (optarg);
	  if (slope_init <= 0)
	    {
	      printf ("slope_init should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 17:
	  thresh_init = atof (optarg);
	  break;
	case 19:
	  precision = atof (optarg);
	  if (precision <= 0)
	    {
	      printf ("precision should be a positive real number.\n");
	      return 0;
	    }
	  break;
	case 'g':
	  graded = 1;
	  break;
	case 'I':
	  icc = 1;
	  response_functions = 1;
	  break;
	case '?':
	  return 0;
	  break;
	default:
	  return 0;
	}
    }

  /* the non optional argument */
  if (optind < argc)
    {
      file_name = argv[optind];
    }
  else
    {
      printf ("Missing file_name.\n");
      printf ("Use -h for help.\n");
      return 0;
    }

  /* set the library variable libirt_verbose */
  libirt_verbose = verbose;

  /* if using kernel or PMMLE enable the display of response functions */
  if (penalized || kernel)
    {
      response_functions = 1;
    }

  /* if using PMMLE check if nbr_quad is a power of 2 */
  if (penalized)
    {
      if (fabs (log (nbr_quad) / log (2.0)
		- (int) (log (nbr_quad) / log (2.0) + 0.5)) > 0.001)
	{
	  printf
	    ("nbr_quad should be a power of 2 when using penalized. Use 16, 32, 64, ... \n");
	  return 0;
	}
    }

  return 1;
}

/**
   \brief Parse the answer key and set the options weights.

   \return 1 for success and 0 for failure.
*/
int options_weights_from_answer_key(char * answer_key, gsl_vector_int * items_pos,
				    gsl_vector_int * nbr_options, gsl_vector * options_weights)
{
  int nbr_item = items_pos->size;
  int str_len;
  int i, o, pos, nbr_option;

  gsl_vector_set_all(options_weights, 0.0);

  if(graded) 
    {
      for(i = nbr_item-1; i >= 0; i--)
	{
	  pos = gsl_vector_int_get(items_pos, i);
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  for( o = 0; o < nbr_option; o++)
	    gsl_vector_set(options_weights, pos+o, o+1.0);
	}
      return 1;
    }

  if(!answer_key) 
    {
      if (verbose > 0)
	printf("No answer key, assuming the firsts options are the goods one !\n");
      for(i = nbr_item-1; i >= 0; i--)
	{
	  pos = gsl_vector_int_get(items_pos, i);
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  gsl_vector_set(options_weights, pos, 1.0);
	}
      return 1;
    }

  str_len = strlen(answer_key);

  if(str_len != nbr_item) 
    {
      printf("The answer key is not of the correct length. It is %d and it should be %d.\n",
	     str_len, nbr_item);
      return 0;
    }

  for(i = nbr_item-1; i >= 0; i--)
    {
      pos = gsl_vector_int_get(items_pos, i);
      nbr_option = gsl_vector_int_get(nbr_options, i);
      answer_key[i+1] = 0;
      o = atoi(&(answer_key[i]));
      if(o < 1 || o > nbr_option)
	{
	  printf("The %dth option in the answer key is outside the valid range.\n", i+1);
	  return 0;
	}
      gsl_vector_set(options_weights, pos+o-1, 1.0);
    }

  return 1;
}

/** \brief The main function of the program. */
int
main (int argc, char *argv[])
{
  int i, k, j, o, nbr_pattern, ret_val, nbr_deg, nbr_option_tot, pos, nbr_option,
    nbr_notconverge = 0;
  double min_weight, max_weight, weight_01, delta;
  FILE *file;
  gsl_matrix_int *patterns=NULL, *patterns_exp=NULL, *patterns_group=NULL;
  gsl_vector *quad_points=NULL, *quad_weights=NULL,
    *slopes=NULL, *thresholds=NULL,
    *slopes_stddev=NULL, *thresh_stddev=NULL,
    *counts=NULL, *abilities=NULL, *abilities_stddev=NULL;
  gsl_vector_int *index=NULL, *nbr_options, *items_pos;
  gsl_vector *options_weights;
  gsl_matrix *probs=NULL, *probs_stddev=NULL, *post;
  gsl_vector_int *ignore=NULL;
  gsl_vector_int *notconverge=NULL;
  gsl_matrix *iccs, *iccs_stddev;

  /* parse the options and arguments */
  if (!parse_arg (argc, argv))
    {
      return 1;
    }

  /* printing the libirt version */
  if (verbose > 0)
    {
      printf ("Using version %s of libirt.\n", libirt_version);
    }

  /* open the file */
  if (NULL == (file = fopen (file_name, "r")))
    {
      printf ("Unable to open %s.\n", file_name);
      return 1;
    }

  /* read it */
  if (!read_mc_patterns_delimited (file, nbr_item, nbr_subject,
				   skip, delimiter, offset, extra,
				   trimming, &patterns, &nbr_options)) return 1;

  /* close it */
  fclose (file);

  nbr_subject = patterns->size1;
  nbr_item = patterns->size2;

  /* print the numbers of patterns and items read */
  if (verbose > 0)
    {
      printf ("%d subjects and %d items read.\n", nbr_subject, nbr_item);
    }

  /* if using PMMLE set the default smoothing factor */
  if (penalized && !smooth_factor)
    {
      smooth_factor = 4 * pow(nbr_subject, 0.2);
    }

  /* if using kernel set the default bandwith */
  if (kernel && !smooth_factor)
    {
      smooth_factor = 2.7 * pow(nbr_subject, -0.2);
    }

  /* count the total number of options */
  nbr_option_tot = 0;
  for(i = 0; i < nbr_item; i++) nbr_option_tot += gsl_vector_int_get(nbr_options, i);

  /* expand the options of each items */
  patterns_exp = gsl_matrix_int_alloc (nbr_subject, nbr_option_tot);
  items_pos = gsl_vector_int_alloc (nbr_item);
  patterns_expand (patterns, nbr_options, patterns_exp, items_pos);

  /* get the options weights */
  options_weights = gsl_vector_alloc(nbr_option_tot);
  if(!options_weights_from_answer_key(answer_key, items_pos,
				      nbr_options, options_weights)) return 1;
  
  /* group the identicals patterns together */
  if (grouping)
    {
      patterns_counts (patterns_exp, &index, &patterns_group, &counts);
      nbr_pattern = patterns_group->size1;
      if (verbose > 0)
	{
	  printf ("%d patterns after grouping.\n", nbr_pattern);
	}
    }
  else
    {
      if (verbose > 0 && nbr_subject / (nbr_option_tot-nbr_item) > 99)
	{
	  printf ("Grouping the patterns might speed up the process.\n");
	}
      patterns_group = patterns_exp;
      nbr_pattern = patterns_group->size1;
      counts = NULL;
    }

  /* generate the quadrature points and weights */
  quad_points = gsl_vector_alloc(nbr_quad);
  quad_weights = gsl_vector_alloc(nbr_quad);
  quadrature (nbr_quad, quad_from, quad_to, quad_points, quad_weights);

  if (verbose > 0)
    printf ("Using %d quadrature points from %.5lg to %.5lg.\n",
	    nbr_quad, quad_from, quad_to);

  /* print the quadrature points and weights */
  if (verbose > 1)
    {
      printf ("The quadrature points and weights :\n");
      printf ("%10s  %10s  %10s\n", "NUM", "POINT", "WEIGHT");
      for (k = 0; k < nbr_quad; k++)
	{
	  printf ("%10d  %10.4lf  %10.4lf\n", k + 1,
		  gsl_vector_get (quad_points, k),
		  gsl_vector_get (quad_weights, k));
	}
      printf ("\n");
    }

  /* allocate the memory for the parameters */
  if (ability || response_functions)
    {
      probs = gsl_matrix_alloc (nbr_option_tot, nbr_quad);
      probs_stddev = gsl_matrix_alloc (nbr_option_tot, nbr_quad);
      gsl_matrix_set_all (probs_stddev, 0.0);
    }
  if (icc)
    {
      iccs = gsl_matrix_alloc (nbr_item+1, nbr_quad);
      iccs_stddev = gsl_matrix_alloc (nbr_item+1, nbr_quad);
    }
  if(!penalized && !kernel)
    {
      thresholds = gsl_vector_alloc(nbr_option_tot);
      thresh_stddev = gsl_vector_alloc(nbr_option_tot);
      if(graded)
	{
	  slopes = gsl_vector_alloc(nbr_item);
	  slopes_stddev = gsl_vector_alloc(nbr_item);
	} else {
	  slopes = gsl_vector_alloc(nbr_option_tot);
	  slopes_stddev = gsl_vector_alloc(nbr_option_tot);
	}
    }

  /* set the initials values */
  if (penalized)
    {
      /* estimate the response functions by kernel */
      nadaraya_watson_mc (2.7 * pow(nbr_subject, -0.2), patterns_exp,
			  items_pos, nbr_options, 
			  options_weights, quad_points, quad_weights, 
			  probs, NULL);
    }
  else if(!kernel)
    {
      for (i = 0; i < nbr_item; i++)
	{
	  pos = gsl_vector_int_get(items_pos, i);
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  min_weight = 1e10;
	  max_weight = -1e10;

	  for (o = 0; o < nbr_option; o++)
	    {
	      if(gsl_vector_get(options_weights,pos+o) < min_weight)
		min_weight = gsl_vector_get(options_weights,pos+o);
	      if(gsl_vector_get(options_weights,pos+o) > max_weight)
		max_weight = gsl_vector_get(options_weights,pos+o);
	    }

	  if(graded)
	    {
	      delta = thresh_init/(nbr_option-2.0);
	      gsl_vector_set(thresholds, pos, -thresh_init);
	      gsl_vector_set(thresholds, pos+ nbr_option-1, thresh_init);
	      for (o = 1; o < nbr_option-1; o++)
		{
		  gsl_vector_set(thresholds, pos+o, -thresh_init+(2*o-1)*delta);
		}
	      gsl_vector_set(slopes, i, slope_init);
	    }
	  else
	    {
	      for (o = 0; o < nbr_option; o++)
		{
		  weight_01 = (gsl_vector_get(options_weights,pos+o) - min_weight)
		    / (max_weight - min_weight);
		  gsl_vector_set(thresholds, pos+o, thresh_init*(2*weight_01-1));
		  gsl_vector_set(slopes, pos+o, slope_init*(2*weight_01-1));
		}
	    }
	}

      /* print the initials values of the parameters */
      if (verbose > 2)
	{
	  printf ("The initial parameters :\n");
	  printf ("%10s %10s %10s %10s\n",
		  "ITEM", "OPTION", "SLOPE", "THRESH");
	  for (i = 0; i < nbr_item; i++)
	    {
	      pos = gsl_vector_int_get(items_pos, i);
	      nbr_option = gsl_vector_int_get(nbr_options, i);
	      for (o = 0; o < nbr_option; o++)
		{
		  printf ("%10d %10d %10.5lf %10.5lf\n", i + 1, o + 1,
			  gsl_vector_get (slopes, pos+(graded?(i-pos):o)),
			  gsl_vector_get (thresholds, pos+o));
		}
	    }
	  printf ("\n");
	}
    }

  /* check for degenerate items */
  ignore = gsl_vector_int_alloc(nbr_item);
  nbr_deg = set_ignore_mc(patterns_exp, nbr_options, items_pos, probs, thresholds, ignore);
  if (verbose > 0 && nbr_deg > 0)
    {
      printf ("There is %d degenerate items.\n", nbr_deg);
    }

  /* reset the convergence flag */
  /* it will be the number of items that didn't converged */
  ret_val = 0;
  notconverge = gsl_vector_int_alloc(nbr_item);

  /* print the description of the estimation method */
  if (verbose > 0)
    {
      if(!kernel)
	{
	  printf ("EM iterations : %d.\n", max_em_iter);
	  printf ("Newton iterations : %d.\n", max_nr_iter);
	  printf ("Precision : %.5lg.\n", precision);
	}
      if (penalized)
	printf ("Using PMMLE with a smooth factor of %.5lg.\n",
		smooth_factor);
      else if (kernel)
	printf ("Using kernel regression with a bandwidth of %.5lg.\n",
		smooth_factor);
      else if (graded)
	printf ("Using the graded model.\n");
      else
	printf ("Using the multivariate logistic model.\n");
      fflush (stdout);
    }

  if (penalized)
    /* estimate the response functions */
    ret_val =
      em_mple_wave_mc (max_em_iter, max_nr_iter, precision, smooth_factor,
		       patterns_group, counts, quad_points, quad_weights,
		       items_pos, nbr_options, 
		       probs, probs_stddev, ignore, 
		       &nbr_notconverge, notconverge, 0);
  else if (kernel)
    /* estimate the response functions */
    nadaraya_watson_mc (smooth_factor, patterns_exp, 
			items_pos, nbr_options, 
			options_weights, quad_points, quad_weights,
			probs, probs_stddev);
  else if (graded)
    {
      /* estimate the parameters */
      ret_val = mmle_2plm_grad (max_em_iter, max_nr_iter, precision,
				patterns_group, counts, quad_points, quad_weights, 
				items_pos, nbr_options,
				thresholds, thresh_stddev, slopes, slopes_stddev,
				ignore, &nbr_notconverge, notconverge, 0);

      if (response_functions)
	{
	  /* compute the response functions */
	  probs_2plm_grad (slopes, thresholds, nbr_options, items_pos, quad_points, probs, NULL);
	}
    }
  else
    {
      /* estimate the parameters */
      ret_val = mmle_2plm_mc (max_em_iter, max_nr_iter, precision,
			      patterns_group, counts, quad_points, quad_weights, 
			      items_pos, nbr_options,
			      thresholds, thresh_stddev, slopes, slopes_stddev,
			      ignore, &nbr_notconverge, notconverge, 0);

      if (response_functions)
	{
	  /* compute the response functions */
	  probs_2plm_mc (slopes, thresholds, nbr_options, items_pos, quad_points, probs);
	}
    }

  /* print a message if some items didn't converged */
  if (verbose > 0 && nbr_notconverge > 0)
    {
      printf ("Warning : %d items didn't converged.\n", ret_val);
      printf ("Run with -VVVV to see the details.\n");
    }

  if (icc)
    {
      /* compute the icc */
      icc_from_probs (probs, probs_stddev, options_weights,
		      nbr_options, items_pos,
		      iccs, iccs_stddev);
    }

  if (response_functions)
    {
      /* print the response functions */
      printf ("%10s", "POINT");
      for (i = 0; i < nbr_item; i++)
	{
	  pos = gsl_vector_int_get(items_pos, i);
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  if (icc) printf ("     ITEM_%02d       SE_%02d", i + 1, i + 1);
	  else for (o = 0; o < nbr_option; o++)
	    printf ("   ITEM_%02d_%01d     SE_%02d_%01d", i + 1, o+1, i + 1, o+1);
	}
      printf ("\n");
      for (k = 0; k < nbr_quad; k++)
	{
	  printf ("%10.5lf", gsl_vector_get (quad_points, k));
	  for (i = 0; i < nbr_item; i++)
	    {
	      pos = gsl_vector_int_get(items_pos, i);
	      nbr_option = gsl_vector_int_get(nbr_options, i);
	      if (icc) printf ("  %10.5lf  %10.3lg",
			       gsl_matrix_get (iccs, i, k),
			       gsl_matrix_get (iccs_stddev, i, k));
	      else for (o = 0; o < nbr_option; o++)
		printf ("  %10.5lf  %10.3lg",
			gsl_matrix_get (probs, pos+o, k),
			gsl_matrix_get (probs_stddev, pos+o, k));
	    }
	  printf ("\n");
	}
    }
  else
    {
      /* print the items parameters */
      printf ("%6s  %6s  %10s  %10s  %10s  %10s\n",
	      "ITEM", "OPTION", "SLOPE", "SLO_SE", "THRESH", "THR_SE");
      for (i = 0; i < nbr_item; i++)
	{
	  pos = gsl_vector_int_get(items_pos, i);
	  nbr_option = gsl_vector_int_get(nbr_options, i);
	  for (o = 0; o < nbr_option; o++)
	    printf
	      ("%6d  %6d  %10.5lf  %10.3lg  %10.5lf  %10.3lg\n",
	       i + 1, o+1, gsl_vector_get (slopes, pos+(graded?(i-pos):o)),
	       gsl_vector_get (slopes_stddev, pos+(graded?(i-pos):o)),
	       gsl_vector_get (thresholds, pos+o), gsl_vector_get (thresh_stddev, pos+o));
	}
    }

  /* estimate the abilities */
  if (ability)
    {
      printf ("\n");

      /* allocate the memory for the abilities */
      abilities = gsl_vector_alloc (nbr_pattern);
      abilities_stddev = gsl_vector_alloc (nbr_pattern);
      post = gsl_matrix_alloc (nbr_pattern, nbr_quad);

      /* estimate the abilities by the bayes expected a posteriori method */
      if (verbose > 0)
	{
	  printf ("Estimating the abilities with prior (EAP).\n");
	  fflush (stdout);
	}
      
      if (!response_functions)
	{
	  /* compute the response functions */
	  if(graded) probs_2plm_grad (slopes, thresholds, nbr_options, items_pos,
				      quad_points, probs, NULL);
	  else probs_2plm_mc (slopes, thresholds, nbr_options, items_pos,
			      quad_points, probs);
	}
      
      posteriors_mc (patterns_group, probs, nbr_options, items_pos, quad_weights, post);

      eap_abilities (post,
		     quad_points, quad_weights, abilities,
		     abilities_stddev);

      /* print the abilities */
      printf ("%10s  %10s  %10s\n", "SUBJECT", "ABILITY", "ABI_SE");
      for (j = 0; j < nbr_subject; j++)
	{
	  if (grouping)
	    k = gsl_vector_int_get (index, j);
	  else
	    k = j;
	  printf ("%10d  %10.5lf  %10.3lg\n", j + 1,
		  gsl_vector_get (abilities, k),
		  gsl_vector_get (abilities_stddev, k));
	}
    }

  /* free the memory */
  gsl_matrix_int_free (patterns);
  gsl_matrix_int_free (patterns_exp);
  gsl_vector_int_free (nbr_options);
  gsl_vector_int_free (items_pos);
  gsl_vector_int_free (ignore);
  gsl_vector_int_free (notconverge);
  gsl_vector_free (options_weights);
  gsl_vector_free (quad_points);
  gsl_vector_free (quad_weights);
  if (ability || response_functions)
    {
      gsl_matrix_free (probs);
      gsl_matrix_free (probs_stddev);
    }
  if (icc)
    {
      gsl_matrix_free (iccs);
      gsl_matrix_free (iccs_stddev);
    }
  if(!penalized && !kernel)
    {
      gsl_vector_free (slopes);
      gsl_vector_free (thresholds);
      gsl_vector_free (slopes_stddev);
      gsl_vector_free (thresh_stddev);
    }
  if (grouping)
    {
      gsl_matrix_int_free (patterns_group);
      gsl_vector_int_free (index);
      gsl_vector_free (counts);
    }
  if (ability)
    {
      gsl_vector_free (abilities);
      gsl_vector_free (abilities_stddev);
      gsl_matrix_free (post);
    }

  return 0;
}
